"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.createRoutesExpress = createRoutesExpress;
const express_1 = __importDefault(require("express"));
const route_handlers_1 = require("../controllers/route_handlers");
const path_1 = __importDefault(require("path"));
/**
 * Creates API routes using Express.js framework.
 * @param filesystem Instance of a IFileSystemService
 * @param jobQueue Instance of a IJobQueue
 * @param settings Instance of a ISettingsService
 * @param config Instance of a IConfigurationService
 * @param index_path Absolute path to index file to serve frontend (index.html)
 * @param logger Instance of a ILogger
 * @returns Created router
 */
function createRoutesExpress(filesystem, jobQueue, settings, config, resources_path, plugins_directory, logger) {
    const router = express_1.default.Router();
    // Capture routes
    router.get('/captures', (req, res) => (0, route_handlers_1.getCapturesList)(req, res, filesystem, logger));
    router.patch('/captures/:basename', (req, res) => (0, route_handlers_1.patchCapture)(req, res, filesystem, logger));
    router.get('/captures/:basename', (req, res) => (0, route_handlers_1.getCaptureByBasename)(req, res, filesystem, logger));
    router.get('/captures/:capture_basename/:event_basename', (req, res) => (0, route_handlers_1.getEventByBasename)(req, res, filesystem, logger));
    router.get('/captures/:capture_basename/:event_basename/:traitId', (req, res) => (0, route_handlers_1.getTraitById)(req, res, filesystem, logger));
    // Job states
    router.post('/jobs/job', (req, res) => (0, route_handlers_1.addJob)(req, res, settings, jobQueue, logger)); // Post a job request to the queue
    router.get('/jobs', (req, res) => (0, route_handlers_1.getJobs)(req, res, jobQueue, logger)); // Get state information about the job queue
    router.get('/jobs/current', (req, res) => (0, route_handlers_1.getJobsStatus)(req, res, filesystem, logger)); // Get state information for the current job
    router.patch('/jobs/current', (req, res) => (0, route_handlers_1.patchJob)(req, res, jobQueue, logger)); // Patch the current job with a JobChangeRequest
    // Plugins routes
    router.get('/plugins', (req, res) => (0, route_handlers_1.listPlugins)(req, res, filesystem, logger));
    router.get('/plugins/:name', (req, res) => (0, route_handlers_1.getPluginManifest)(req, res, filesystem, logger));
    // Settings routes
    router.post('/settings', (req, res) => (0, route_handlers_1.setSettings)(req, res, settings, logger));
    router.get('/settings', (req, res) => (0, route_handlers_1.getSettings)(req, res, settings, logger));
    // Configuration routes
    router.get('/configuration/:domain', (req, res) => (0, route_handlers_1.getDomainConfigurations)(req, res, config));
    router.patch('/configuration/:domain', (req, res) => (0, route_handlers_1.patchConfig)(req, res, config, logger));
    router.get('/configuration/:domain/:name/:is_default', (req, res) => (0, route_handlers_1.getConfiguration)(req, res, config));
    const root_directory = path_1.default.join(resources_path, 'dist', 'frontend');
    // Serve index
    router.get('/', (req, res) => (0, route_handlers_1.serveIndex)(req, res, root_directory));
    //Helpdesk routes
    router.get('/help', (req, res) => (0, route_handlers_1.serveIndex)(req, res, root_directory, "README.md")); //NOTE: this file is located under client/public/README.md
    router.get('/help/plugins/:name', (req, res) => (0, route_handlers_1.serveIndex)(req, res, plugins_directory, "README.md"));
    return router;
}
//# sourceMappingURL=routes.js.map