## How to use Personal Scroller

1. Make sure it can see the plugins: Personal Scroller uses a plugin system to handle many different SFV platforms efficiently. In order to do this it needs to know where the plugins are on your computer. By default, this is `~/Documents/Personal Scroller/Plugins`

2. How to install plugins: You should download plugins.zip from wherever you downloaded this app and unzip it, then move the individual files (not the plugins folder itself) to `~/Documents/Personal Scroller/Plugins`. If you want, you can also set the plugins path somewhere else under `Settings`.

3. Once plugins are installed, you should see some "Controllers" and some "Operators". What are they?

## [Controllers](#controllers)

Personal Scroller uses Controllers to define what 'traits' to gather from short form videos it sees (for example, `top-comment` or `likes`). 

By default, they are 'dumb' controllers which simply contain a list of traits (you can see which by clicking on the name of the controller)

There's work ongoing for a 'smart' programmable controller that would be able to react to what it sees onscreen. If you want to help, hit me up at [hudmarc](https://github.com/hudmarc) or Pierre at GitLab: [periode](https://gitlab.com/periode)


## [Operators](#operators)

Personal Scroller uses an Operator to *operate* the SFV platform and gather the traits requested by the controllers.

The operator uses Puppeteer under the hood, a website testing framework, and uses it to automatically interact with whatever short-form platform you decide to gather info from.

## Your First Scroll

So, you have some controllers selected and you have one operator picked. What now?

Click 'Start scrolling' to start scrolling!

You can monitor the progress of the scroll by checking the "Status" box. This basically tells you what the current job is doing.

When you're done scrolling, you can click "Halt" in the status box. This will stop the current scroll as soon as possible. You can also force-quit Personal Scroller, but then you might lose the most recent video data.

# Advanced

## How does it work?

Personal Scroller manages splits the concerns of scrolling an SFV platform into a few modular pieces- "what do I want?" becomes traits from controllers, "how do I interact with a specific SFV platform" becomes an operator, and all output is written to strucured JSON to your captures folder.

Under the hood it uses a path optimizer to reduce the number of clicks on onscreen elements, so for example if you request "comments" then "top comment" it doesn't have to open the comments pane, scan the text, close it, open it again and so on, instead it knows it just needs to open the comments panel once and then do what it needs to do there.

This saves time and resources, and it reduces the number of clicks puppeteer needs to do its job.

Do you want to build your own module?

# How do I write a controller?

All plugins are built on a common interface. If you want to build your own, clone the [GitLab repository](https://gitlab.com/periode/personal-scroller) and check out the sample plugins. For example, here is an incredibly basic controller:

```
import { Controller, ControllerContext } from "#sfvc_types/plugin";

/**
 * Super basic controller
 */
export async function activate(context: ControllerContext): Promise<Controller> {
    return {
        getTraits: () => {
            return ["id","url"]
        }
    }
}

```

What does it do?

It returns an object which defines "getTraits", which is a function that returns "id" and "url". That's it, not too difficult.

Why all the extra stuff on top? The controller context contains commmon functions that a plugin might need to do its job. If you clone this repo, VSCode should tell you what you can do with the Context.

Here's a more advanced example:

```
import { Validator, Controller, ControllerContext } from "#sfvc_types/plugin";

export async function activate(context: ControllerContext): Promise<Controller> {
    
    // This validates the string sent to the backend from the frontend. 
    // For example, if you wanted to gather all videos with a certain average color,
    // you could define a configuration called "my_controller":"my_color"
    // and then write a Validator that checks if the color received from the frontend is valid

    const stringValidator: Validator<string> = {
        hint: "Traits to be collected, separated by comma",
        //TODO rewrite this so it actually validates traits!
        validate: function (value: string): boolean {
            if (value.includes(";"))
                return false;
            if (value.includes(" "))
                return false;
            if (value.length === 0)
                return false
            return true;
        }
    }
    
    // This isn't really implemented yet on the frontend,
    // but the idea is that you can configure the settings for a run and then have repeatable runs.
    // What this does is add a configuration called "basic_controller":"traits" which can
    // then be set by the frontend. It uses the validator we defined above.
    context.configurationServer.addConfig(
        "basic_controller",
         "traits", 
         "id,url,creator,created_at,description,likes_count,views_count,comments_count,comments",
         stringValidator)
    return {
        getTraits: () => {
            //If the configuration doesn't exist, return no traits
            if (!context.configurationServer.hasConfig("basic_controller", "traits")) {
                return [];
            }
            //If the configuration hasn't been edited by someone using the API, this will be all the traits this controller supports.
            return context.configurationServer.getConfig("basic_controller", "traits")!.split(",")
        }
    }
}
```

So why is this so complicated?

The idea is that controllers can be reactive: if you wanted to, for example, simulate a certain kind of user or simply like only videos that mention the word "banana", you could do that with a custom controller. Just implement some custom logic on getTraits, and it will be checked before every scroll.

The posibilities are pretty much endless.

If you want more technical info, check out [the developer wiki](https://gitlab.com/periode/personal-scroller/-/wikis/home) (wip!)