"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.randomPromise = randomPromise;
const operation_optimizer_1 = require("../utils/operation_optimizer");
/**
 * SFVC app class.
 */
class SFVC {
    constructor(configurationService, jobQueue, filesystem, logger, puppeteer, job_logger) {
        this.configurationService = configurationService;
        this.jobQueue = jobQueue;
        this.filesystem = filesystem;
        this.logger = logger;
        this.puppeteer = puppeteer;
        this.job_logger = job_logger;
        this.operator_name = "";
        this.operator = undefined;
        this.controllers = new Map;
    }
    async processTraits(traitSet, job) {
        if (!this.operator) {
            return undefined;
        }
        // Builds a trait graph here using traits passed from controller and dependencies from operators and traverse it in-order
        // TODO: See page 40 of SFV architecture documentation on the tablet for more in-depth explanation, put this on the wiki!
        const processed = new operation_optimizer_1.OperationOptimizer(traitSet, this.operator.operations);
        for (const warning of processed.getWarnings()) {
            this.logger.warn(warning);
        }
        const results = new Map();
        for (const operation of processed.getOrderedOperations()) {
            try {
                await operation[1](results);
                await randomPromise();
            }
            catch (error) {
                this.logger.warn(`Failed to gather trait ${operation[0]}: ${error}`);
            }
        }
        this.filesystem.writeEvent(job.request.name, results);
        return results;
    }
    // called on job initialization
    async onJob(job) {
        this.logger.info(`Controllers: ${Array.from(this.controllers.values())} Operator: ${JSON.stringify(this.operator)} Events: ${job.request.events}`);
        try {
            const puppeteerInstance = await this.puppeteer.getInstance();
            for (const controller of job.request.controllers) {
                if (controller.includes("\n")) {
                    throw new Error(`Controller ${controller} failed to load. Newlines are not permitted in controller names.`);
                }
                if (!this.controllers.has(controller)) {
                    let activate;
                    try {
                        const plugin = await this.filesystem.getPlugin(controller, "controller");
                        activate = plugin.activate;
                        if (!activate) {
                            throw new Error(`Controller ${controller} failed to load. Controller is undefined.`);
                        }
                    }
                    catch (error) {
                        throw new Error(`Controller ${controller} failed to load: ${error.message}`);
                    }
                    try {
                        if (activate) {
                            const ctx = {
                                configurationServer: this.configurationService,
                                capture: job.capture,
                                logger: this.job_logger
                            };
                            this.controllers.set(controller, await activate(ctx));
                        }
                    }
                    catch (error) {
                        throw new Error(`Error activating controller ${controller}:\n${error}`);
                    }
                }
            }
            if (this.operator_name != job.request.operator) {
                try {
                    const context = {
                        configurationServer: this.configurationService,
                        capture: job.capture,
                        logger: this.job_logger
                    };
                    const plugin = (await this.filesystem.getPlugin(job.request.operator, "operator"));
                    const activate = plugin.activate;
                    this.operator = await activate(context, puppeteerInstance.page);
                }
                catch (error) {
                    throw new Error(`Error activating operator ${job.request.operator}:\n${error}`);
                }
            }
            if (!this.operator) {
                job.error = `Operator ${job.request.operator} failed to load.`;
                throw new Error("failed to get plugin operator");
            }
            //TODO: seems like this doesn't load??
            for (let index = 0; index < job.request.events; index++) {
                const traitSet = new Set();
                // If no operator was present, return early.
                if (!this.operator) {
                    return;
                }
                for (const controller of this.controllers.values()) {
                    try {
                        controller.getTraits().forEach(trait => traitSet.add(trait));
                    }
                    catch (error) {
                        throw new Error(`Error on getTraits() in controller ${controller}:\n${error}`);
                    }
                }
                this.logger.info(`Processing traits ${Array.from(traitSet.keys())}`);
                const results = await this.processTraits(traitSet, job).catch(err => {
                    throw new Error(`Error in operator:\n${err}`);
                });
                //If you want the full trait set to persist then remove this. TODO: maybe this should be a config that controllers can expose?
                traitSet.clear();
                if (results) {
                    for (const controller of this.controllers.values()) {
                        if (controller.onTrait) {
                            for (const trait of results.keys()) {
                                const requestedTraits = await controller.onTrait(trait, results).catch(err => {
                                    throw new Error(`Error in controller on Trait '${trait}':\n${err}`);
                                }); //controllers should be able to add more traits to the TraitSet here.
                                if (requestedTraits) {
                                    for (const trait of requestedTraits) {
                                        traitSet.add(trait);
                                    }
                                }
                            }
                        }
                    }
                }
                if (traitSet.size > 0) {
                    //NOTE: if you want to react to content in an SFV more than once this will have to be some kind of loop.
                    //For now this just runs if any additional traits were requested after the main collection event.
                    await this.processTraits(traitSet, job).catch(err => {
                        throw new Error(`Error in operator:\n${err}`);
                    });
                    ;
                }
                await this.operator.scroll().catch(err => {
                    throw new Error(`Error in operator scroll function:\n${err}`);
                });
            }
        }
        catch (error) {
            job.error = error.message;
            this.logger.error(`Error on job: \n${error.stack}`);
        }
        this.logger.info("Job is complete");
        this.jobQueue.markComplete(job); // TODO: better mechanism for job proccessing, should automatically mark a job as failed if there is an exception during its processing
        await this.puppeteer.dispose();
    }
}
exports.default = SFVC;
function randomPromise() {
    return () => new Promise(r => setTimeout(r, 1000 + (Math.random() * 1000)));
}
