"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.serveIndex = exports.getSettings = exports.setSettings = exports.getPluginManifest = exports.getJobsStatus = exports.getJobs = exports.addJob = exports.getTraitById = exports.getEventById = exports.getCaptureById = exports.patchCapture = exports.getCapturesList = exports.listPlugins = void 0;
const schemas_1 = require("../../../schemas/schemas");
const uuid_1 = require("uuid");
/**
 * List all available plugin names.
 *
 * @route GET /plugins
 * @returns {Array<string>|Error(500)} - List of plugin names or 500 error on failure.
 */
const listPlugins = async (req, res, filesystem, logger) => {
    try {
        const plugins = await filesystem.listPlugins(); //needs to be passed in somehow
        res.send(plugins);
    }
    catch (error) {
        logger.error(`Failed to list plugins: ${error} ${error.stack}`);
        res.status(500).send(error.message);
    }
};
exports.listPlugins = listPlugins;
/**
 * Retrieve list of all available captures.
 *
 * @route GET /captures
 * @returns {Array<Capture>|Error(500)} - List of capture metadata or 500 error.
 */
const getCapturesList = async (req, res, filesystem, logger) => {
    try {
        // Fetch list of captures
        res.json(await filesystem.getCaptures());
    }
    catch (error) {
        logger.error(`Failed to list captures: ${error}`);
        res.status(500).send(error.message);
    }
};
exports.getCapturesList = getCapturesList;
/**
 * Update metadata for a given capture.
 *
 * @route PATCH /captures/:captureId
 * @returns {Status(200)|Error(500)} - 200 on success, 500 on validation or server error.
 */
const patchCapture = async (req, res, filesystem, logger) => {
    try {
        const capture = schemas_1.PatchCaptureActionSchema.parse(req.body);
        const { captureSlug } = req.params;
        await filesystem.patchCapture(captureSlug, capture.name);
        res.sendStatus(200);
    }
    catch (error) {
        logger.error(`Failed to patch capture: ${error}`);
        res.status(500).send(error.message);
    }
};
exports.patchCapture = patchCapture;
/**
 * Get a specific capture by ID.
 *
 * @route GET /captures/:captureId
 * @returns {Capture|Status(404)|Error(500)} - Capture object, 404 if not found, 500 on error.
 */
const getCaptureById = async (req, res, filesystem, logger) => {
    const { captureId } = req.params;
    try {
        const capture = await filesystem.getCapture(captureId);
        if (capture)
            res.json(capture);
        else
            res.sendStatus(404);
    }
    catch (error) {
        logger.error(`Failed to get capture ${captureId}: ${error}`);
        res.status(500).send(error.message);
    }
};
exports.getCaptureById = getCaptureById;
/**
 * Get a specific event from a capture by its ID.
 *
 * @route GET /captures/:captureId/events/:eventId
 * @returns {CollectedEvent|Status(404)|Error(500)} - Event object or 404/500.
 */
const getEventById = async (req, res, filesystem, logger) => {
    const { captureId, eventId } = req.params;
    try {
        const event = await filesystem.getEvent(captureId, eventId);
        if (event)
            res.json(event);
        else
            res.sendStatus(404);
    }
    catch (error) {
        logger.error(`Failed to get event ${eventId} (capture: ${captureId}): ${error}`);
        res.status(500).send(error.message);
    }
};
exports.getEventById = getEventById;
/**
 * Retrieve a gathered trait by ID from an event in a capture.
 *
 * @route GET /captures/:captureId/events/:eventId/traits/:traitId
 * @returns {GatheredTrait|Status(404)|Error(500)} - Trait data or 404/500.
 */
const getTraitById = async (req, res, filesystem, logger) => {
    try {
        const { captureId, eventId, traitId } = req.params;
        const trait = await filesystem.getGatheredTrait(captureId, eventId, traitId);
        if (trait)
            res.json(trait);
        else
            res.sendStatus(404);
    }
    catch (error) {
        logger.error(`Failed to get trait: ${error}`);
        res.status(500).send(error.message);
    }
};
exports.getTraitById = getTraitById;
/**
 * Adds job to job queue.
 *
 * @route POST /app/jobs
 * @returns {Status(202)|Error(500)} - 202 on success, 500 on error.
 */
const addJob = async (req, res, formatter, settings, jobQueue, logger) => {
    logger.debug("adding job");
    try {
        req.body.id = (0, uuid_1.v7)(); // internal job ID
        const job_request = schemas_1.JobRequestSchema.parse(req.body);
        const formatted_name = await formatter.format_capture((await settings.getSettings()).event_format, job_request.id);
        jobQueue.addJob(job_request, formatted_name);
        res.sendStatus(202);
    }
    catch (error) {
        logger.error(`Failed to add job: ${error}`);
        res.status(500).send(error.message);
    }
};
exports.addJob = addJob;
/**
 * Get job queue state.
 *
 * @route GET /app/jobs
 * @returns {{ status: Array<Job> }|Error(500)} - Application job queue state or 500 error.
 */
const getJobs = async (req, res, jobQueue, logger) => {
    try {
        // Return app state + debug information
        res.json({ current: jobQueue.getCurrentJob(), count: jobQueue.getCount() });
    }
    catch (error) {
        logger.error(error.message);
        res.status(500).send(error.message);
    }
};
exports.getJobs = getJobs;
/**
 * Get the status of the current running job.
 *
 * @route GET /app/jobs/status
 * @returns {string|Error(500)} - Job status as an unparsed JSON string or 500 error.
 */
const getJobsStatus = async (req, res, filesystem, logger) => {
    try {
        const log = await filesystem.getJobLog();
        res.json(log);
    }
    catch (error) {
        logger.error(error.message);
        res.status(500).send(error.message);
    }
};
exports.getJobsStatus = getJobsStatus;
/**
 * Get the manifest for a given plugin by name.
 *
 * @route GET /plugins/:name/manifest
 * @returns {object|Error(500)} - Plugin manifest JSON or 500 error.
 */
const getPluginManifest = async (req, res, filesystem, logger) => {
    try {
        const { name } = req.params;
        const plugin = await filesystem.getPluginManifest(name);
        res.json(plugin);
    }
    catch (error) {
        logger.error(error.message);
        res.status(500).send(error.message);
    }
};
exports.getPluginManifest = getPluginManifest;
/**
 * Set app settings
 *
 * @route PUT /settings/
 * @returns {Status(204)|Error(500)} - 204 on success, 500 on error.
 */
const setSettings = async (req, res, settings, logger) => {
    logger.info("Setting app settings.");
    try {
        settings.setSettingsWithError(req.body);
        settings.saveSettings();
        res.sendStatus(204);
    }
    catch (error) {
        logger.error(error.message);
        res.status(500).send(error.message);
    }
};
exports.setSettings = setSettings;
/**
 * Get global app settings.
 *
 * @route GET /settings
 * @returns {object|Error(500)} - App settings JSON or 500 error.
 */
const getSettings = async (req, res, settings, logger) => {
    try {
        res.json(await settings.getSettings());
    }
    catch (error) {
        logger.error(error.message);
        res.status(500).send(error.message);
    }
};
exports.getSettings = getSettings;
/**
 * Serve the client-side application entry point.
 *
 * @route GET /*
 * @returns {HTML} - Sends the index HTML file.
 */
const serveIndex = async (req, res, index) => {
    res.sendFile(index);
};
exports.serveIndex = serveIndex;
