"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * The configuration service implements common methods for configuring the app's modules.
 * TODO: implement reactivity so that, for example, plugins can respond to live configuration changes
 */
class ConfigurationService {
    constructor() {
        this.configMaps = {
            string: new Map(),
            number: new Map(),
            boolean: new Map(),
        };
        this.configValidators = {
            string: new Map(),
            number: new Map(),
            boolean: undefined,
        };
    }
    addConfig(plugin, name, value, validator) {
        const type = typeof value;
        const configMap = this.configMaps[type];
        if (validator) {
            const validatorMap = this.configValidators[type];
            if (type == 'boolean')
                throw "Validators are not allowed on booleans";
            if (!validatorMap.has(plugin)) {
                validatorMap.set(plugin, new Map());
            }
            validatorMap.get(plugin).set(name, validator);
        }
        if (!configMap.has(plugin)) {
            configMap.set(plugin, new Map());
        }
        configMap.get(plugin).set(name, value);
    }
    setConfig(plugin, name, value) {
        for (const type of ["string", "number", "boolean"]) {
            if (type == typeof value) {
                const map = this.configMaps[type].get(plugin);
                if (map) {
                    if (type != 'boolean') {
                        const validators = this.configValidators[type].get(plugin);
                        const validator = validators?.get(name);
                        if (validator) {
                            if (!validator.validate(value)) {
                                return false;
                            }
                        }
                    }
                    map.set(name, value);
                }
                return true;
            }
        }
        return false;
    }
    getConfig(plugin, name) {
        for (const type of ["string", "number", "boolean"]) {
            const value = this.configMaps[type].get(plugin)?.get(name);
            if (value !== undefined)
                return value;
        }
        return undefined;
    }
    hasConfig(plugin, name) {
        return this.isString(plugin, name) || this.isNumber(plugin, name) || this.isBool(plugin, name);
    }
    isString(plugin, name) {
        const found = this.configMaps.string.get(plugin);
        if (found && found.has(name)) {
            return true;
        }
        return false;
    }
    isNumber(plugin, name) {
        const found = this.configMaps.number.get(plugin);
        if (found && found.has(name)) {
            return true;
        }
        return false;
    }
    isBool(plugin, name) {
        const found = this.configMaps.boolean.get(plugin);
        if (found && found.has(name)) {
            return true;
        }
        return false;
    }
}
exports.default = ConfigurationService;
