"use strict";
var __createBinding = (this && this.__createBinding) || (Object.create ? (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    var desc = Object.getOwnPropertyDescriptor(m, k);
    if (!desc || ("get" in desc ? !m.__esModule : desc.writable || desc.configurable)) {
      desc = { enumerable: true, get: function() { return m[k]; } };
    }
    Object.defineProperty(o, k2, desc);
}) : (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    o[k2] = m[k];
}));
var __setModuleDefault = (this && this.__setModuleDefault) || (Object.create ? (function(o, v) {
    Object.defineProperty(o, "default", { enumerable: true, value: v });
}) : function(o, v) {
    o["default"] = v;
});
var __importStar = (this && this.__importStar) || (function () {
    var ownKeys = function(o) {
        ownKeys = Object.getOwnPropertyNames || function (o) {
            var ar = [];
            for (var k in o) if (Object.prototype.hasOwnProperty.call(o, k)) ar[ar.length] = k;
            return ar;
        };
        return ownKeys(o);
    };
    return function (mod) {
        if (mod && mod.__esModule) return mod;
        var result = {};
        if (mod != null) for (var k = ownKeys(mod), i = 0; i < k.length; i++) if (k[i] !== "default") __createBinding(result, mod, k[i]);
        __setModuleDefault(result, mod);
        return result;
    };
})();
Object.defineProperty(exports, "__esModule", { value: true });
const fs_1 = require("fs");
const path = __importStar(require("path"));
const schemas_1 = require("../schemas/schemas");
//TODO: Move capture creation logic from job_queue to here! (see line 16 of JobQueue)
/**
 * A service class for managing file system interactions related to captures and plugins.
 */
class FileSystemService {
    /**
     *
     * @param formatter
     * @param settings
     */
    /**
     * Constructs the file system service with a formatting utility. Automatically registers this FileSystemService object with the settings service.
     * @param formatter - An instance of the formatting service.
     * @param settings - An instance of the settings service. NOTE: This constructor automatically calls `settings.registerFilesystem(this)`
     * @param logger  - The logger to use with this filesystem service.
     * @param data_directory  - The data directory associated with this filesystem service.
     */
    constructor(formatter, settings, logger, data_directory) {
        this.formatter = formatter;
        this.settings = settings;
        this.logger = logger;
        this.data_directory = data_directory;
        settings.registerFilesystem(this);
    }
    /**
     * Loads a plugin's manifest and validates it. TODO: load package.json instead of manifest.json?
     * @param name - The name of the plugin (folder name).
     * @returns A promise resolving to the validated plugin manifest.
     */
    async getPluginManifest(name) {
        const manifestPath = await this.getPath(name, "manifest.json");
        const content = await fs_1.promises.readFile(manifestPath, "utf-8");
        const parsed = JSON.parse(content);
        return schemas_1.PluginManifestSchema.parse(parsed);
    }
    /**
    * Gets the current job log.
    * @returns A promise resolving to the current job log.
    */
    async getJobLog() {
        const logPath = path.join(this.data_directory, 'job.log');
        return await fs_1.promises.readFile(logPath, "utf-8");
    }
    /**
     * Gets the path for a plugin file
     * @param plugin_name The plugin name
     * @param filename The file to retrieve
     * @returns The path.
     */
    async getPath(plugin_name, filename) {
        return path.join(await this.settings.getPluginsDirectory(), plugin_name, filename);
    }
    /**
     * Returns a plugin loaded from the disk. TODO: Load index file correctly.
     * @param name Plugin to load.
     * @param type Optional: Expected plugin type.
     */
    async getPlugin(name, type) {
        const manifest = await this.getPluginManifest(name);
        if (type && manifest.type !== type)
            throw Error("Plugin is of incorrect type");
        const pluginPath = await this.getPath(name, "index.js");
        const module = await import(path.win32.resolve(pluginPath));
        const { activate } = module;
        return { activate: activate, manifest: manifest };
    }
    /**
     * Loads a gathered trait's raw HTML and scraped JSON.
     * @param captureId - ID of the capture.
     * @param eventId - ID of the event.
     * @param traitId - ID of the trait.
     * @returns A promise resolving to the gathered trait object.
     */
    async getGatheredTrait(captureId, eventId, traitId) {
        const basePath = path.join(await this.settings.getCapturesDirectory(), captureId, eventId, traitId);
        const rawPath = path.join(basePath, `raw_${traitId}.html`);
        const scrapedPath = path.join(basePath, `scraped_${traitId}.json`);
        const raw = await fs_1.promises.readFile(rawPath, "utf-8");
        const scrapedContent = await fs_1.promises.readFile(scrapedPath, "utf-8");
        const scraped = JSON.parse(scrapedContent);
        return schemas_1.GatheredTraitSchema.parse({ raw, scraped });
    }
    /**
     * Loads an event's manifest and all trait JSON files.
     * @param captureId - ID of the capture.
     * @param eventId - ID of the event.
     * @returns A promise resolving to the event object.
     */
    async getEvent(captureId, eventId) {
        const basePath = path.join(await this.settings.getCapturesDirectory(), captureId, eventId);
        const manifestContent = await fs_1.promises.readFile(path.join(basePath, "manifest.json"), "utf-8");
        const manifest = JSON.parse(manifestContent);
        const traitFiles = await fs_1.promises.readdir(basePath);
        const traitJsonFiles = traitFiles.filter((f) => /^traits(_\d+)?\.json$/.test(f));
        const traits = [];
        for (const file of traitJsonFiles) {
            const traitContent = await fs_1.promises.readFile(path.join(basePath, file), "utf-8");
            const parsed = JSON.parse(traitContent);
            traits.push(...parsed);
        }
        return { manifest, traits };
    }
    async writeEvent(capture_id, results) {
        const settings = await this.settings.getSettings();
        const captureDir = capture_id.length < 1 ? settings.captures_directory : path.join(settings.captures_directory, capture_id);
        await fs_1.promises.mkdir(captureDir, { recursive: true });
        const existingEventDirs = await fs_1.promises.readdir(captureDir);
        // Create a timestamped event subfolder
        const eventDir = path.join(settings.captures_directory, `video_${existingEventDirs.length}_${Date.now()}`);
        await fs_1.promises.mkdir(eventDir, { recursive: true });
        // write all scraped traits in single json
        const traits = new Map();
        for (const [trait, { processed }] of results.entries()) {
            const safeTrait = trait.replace(/[<>:"/\\|?*]+/g, '_'); // optional sanitization
            traits.set(safeTrait, processed);
        }
        await fs_1.promises.writeFile(path.join(eventDir, `scraped_traits.json`), JSON.stringify(Object.fromEntries(traits)), 'utf-8');
        // Iterate over raw traits and write files
        for (const [trait, { raw }] of results.entries()) {
            const safeTrait = trait.replace(/[<>:"/\\|?*]+/g, '_'); // optional sanitization
            const rawFile = path.join(eventDir, `raw_${safeTrait}.html`);
            if (raw.length > 0)
                await fs_1.promises.writeFile(rawFile, raw, 'utf-8');
        }
        this.logger.info(`written event to: ${path.basename(eventDir)}`);
    }
    async writeCapture(job_request) {
        // Create capture dir
        const captureDir = path.join(await this.settings.getCapturesDirectory(), job_request.id);
        this.logger.debug(`making capture dir: ${captureDir}`);
        await fs_1.promises.mkdir(captureDir, { recursive: true });
        await fs_1.promises.writeFile(path.join(captureDir, "job.json"), JSON.stringify(job_request));
        this.logger.info(`added job: ${job_request.name}`);
    }
    /**
     * Retrieves a list of event folder names for a given capture.
     * @param capture_id - ID of the capture.
     * @returns A promise resolving to a Capture object containing event IDs.
     */
    async getCapture(capture_id) {
        const capturePath = path.join(await this.settings.getCapturesDirectory(), capture_id);
        const entries = await fs_1.promises.readdir(capturePath, { withFileTypes: true });
        const events = entries.filter((entry) => entry.isDirectory()).map((dir) => dir.name);
        return { events };
    }
    /**
     * Renames a capture folder.
     * @param captureId - The current ID/name of the capture folder.
     * @param name - The new name for the folder.
     */
    async patchCapture(captureSlug, new_name) {
        if (!new_name)
            throw new Error("New name is required to rename capture");
        const settings = await this.settings.getSettings();
        const capturePath = path.join(settings.captures_directory, captureSlug);
        const manifest = JSON.parse(await fs_1.promises.readFile(capturePath, "utf-8"));
        manifest.name = new_name;
        await fs_1.promises.writeFile(capturePath, JSON.stringify(manifest));
    }
    /**
     * Lists all available plugin folders.
     * @returns A promise resolving to an array of plugin folder names.
     */
    async listPlugins() {
        this.logger.info("listing plugins");
        const pluginsPath = (await this.settings.getSettings()).plugins_directory;
        // try {
        //     await fs.access(pluginsPath);
        // } catch {
        //     this.logger.warn("plugins dir does not exist, creating...")
        //     await fs.mkdir(pluginsPath, { recursive: true });
        // }
        const entries = await fs_1.promises.readdir(pluginsPath, { withFileTypes: true });
        return entries.filter((entry) => entry.isDirectory()).map((dir) => dir.name);
    }
    /**
     * Lists all available capture folders.
     * @returns A promise resolving to an array of capture folder names.
     */
    async getCaptures() {
        this.logger.info("getting captures");
        const capturesPath = await this.settings.getCapturesDirectory();
        try {
            await fs_1.promises.access(capturesPath);
        }
        catch {
            this.logger.warn("captures dir does not exist, creating...");
            await fs_1.promises.mkdir(capturesPath, { recursive: true });
        }
        const entries = await fs_1.promises.readdir(capturesPath, { withFileTypes: true });
        const captures = [];
        for (const entry of entries) {
            if (entry.isDirectory()) {
                try {
                    await fs_1.promises.access(path.join(capturesPath, entry.name, "manifest.json"));
                    captures.push(entry.name);
                }
                catch {
                    this.logger.error(`could not open manifest.json for: ${entry.name}`);
                }
            }
        }
        return captures;
    }
    async getSettings(settings_directory) {
        try {
            const content = await fs_1.promises.readFile(settings_directory, "utf-8");
            console.log(content);
            const parsed = schemas_1.AppSettingsSchema.parse(JSON.parse(content));
            return parsed;
        }
        catch (raw_error) {
            const error = raw_error;
            //FIXME: too much code duplication here
            if (error.code === "ENOENT") {
                console.log(`Creating default settings at ${this.data_directory}`);
                // Settings file doesn't exist – create a default one
                const blankSettings = this.settings.getDefaultSettings();
                //Initialize the data directory
                await fs_1.promises.mkdir(this.data_directory, { recursive: true });
                //Initialize the settings file
                await fs_1.promises.writeFile(settings_directory, JSON.stringify(blankSettings, null, 2));
                //Initialize the Plugins folder
                await fs_1.promises.mkdir(blankSettings.plugins_directory, { recursive: true });
                //Initialize the Captures folder
                await fs_1.promises.mkdir(blankSettings.captures_directory, { recursive: true });
                return blankSettings;
            }
            // Invalid format or parse error
            console.error("Failed to load settings. File may be corrupted:", error.message);
            const settings_path = settings_directory;
            const timestamp = new Date().toISOString().replace(/[:.]/g, "-");
            const corruptedPath = settings_path.replace(/\\.json$/, "") + `_${timestamp}_corrupted.json`;
            await fs_1.promises.rename(settings_path, corruptedPath);
            const blankSettings = this.settings.getDefaultSettings();
            await fs_1.promises.writeFile(settings_path, JSON.stringify(blankSettings, null, 2));
            return blankSettings;
        }
    }
    async saveSettings(settings_directory, settings) {
        await fs_1.promises.writeFile(settings_directory, JSON.stringify(settings, null, 2), "utf-8");
    }
}
exports.default = FileSystemService;
