"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.JobQueue = void 0;
class JobQueue {
    constructor(jobListeners, logger, filesystem) {
        this.jobListeners = jobListeners;
        this.logger = logger;
        this.filesystem = filesystem;
        this.queue = [];
        this.currentJob = null;
    }
    // Adds the job to the job queue from the request.
    async addJob(request, capture) {
        const job = {
            request: request,
            capture: capture,
            started_at: Date.now()
        };
        this.filesystem.writeCapture(request);
        if (!this.currentJob) {
            this.currentJob = job;
            this.processJob(job);
        }
        else {
            this.queue.push(job);
        }
        return job;
    }
    /**
     * Adds a new listener to the job queue.
     * The listener must be an object with a method `onJob(job: Job): Promise<void>`.
     *
     * @param listenerInstance - An object that implements the `onJob(job: Job)` method.
     */
    addListener(listenerInstance) {
        if (typeof listenerInstance.onJob === 'function') {
            this.jobListeners.add(listenerInstance.onJob.bind(listenerInstance));
        }
        else {
            throw new Error("Listener must implement an 'onJob(job: Job): Promise<void>' method.");
        }
    }
    // Marks the job passed in as complete.
    markComplete(job) {
        if (this.currentJob && this.currentJob === job) {
            this.currentJob = this.queue.shift() || null;
            if (this.currentJob) {
                this.processJob(this.currentJob);
            }
        }
    }
    // Gets the current active job
    getCurrentJob() {
        return this.currentJob;
    }
    // Gets the number of jobs in the queue including the current job
    getCount() {
        return (this.currentJob ? 1 : 0) + this.queue.length;
    }
    async processJob(job) {
        for (const listener of this.jobListeners) {
            try {
                this.logger.info(`awaiting function ${listener.name}`);
                await listener(job);
            }
            catch (error) {
                job.error = error instanceof Error ? error.message : String(error);
                this.logger.error(`error processing job: ${error}`);
            }
        }
    }
}
exports.JobQueue = JobQueue;
