"use strict";
var __createBinding = (this && this.__createBinding) || (Object.create ? (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    var desc = Object.getOwnPropertyDescriptor(m, k);
    if (!desc || ("get" in desc ? !m.__esModule : desc.writable || desc.configurable)) {
      desc = { enumerable: true, get: function() { return m[k]; } };
    }
    Object.defineProperty(o, k2, desc);
}) : (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    o[k2] = m[k];
}));
var __setModuleDefault = (this && this.__setModuleDefault) || (Object.create ? (function(o, v) {
    Object.defineProperty(o, "default", { enumerable: true, value: v });
}) : function(o, v) {
    o["default"] = v;
});
var __importStar = (this && this.__importStar) || (function () {
    var ownKeys = function(o) {
        ownKeys = Object.getOwnPropertyNames || function (o) {
            var ar = [];
            for (var k in o) if (Object.prototype.hasOwnProperty.call(o, k)) ar[ar.length] = k;
            return ar;
        };
        return ownKeys(o);
    };
    return function (mod) {
        if (mod && mod.__esModule) return mod;
        var result = {};
        if (mod != null) for (var k = ownKeys(mod), i = 0; i < k.length; i++) if (k[i] !== "default") __createBinding(result, mod, k[i]);
        __setModuleDefault(result, mod);
        return result;
    };
})();
Object.defineProperty(exports, "__esModule", { value: true });
const path = __importStar(require("path"));
const schemas_1 = require("../schemas/schemas");
const zod_1 = require("zod");
//TODO: Move all filesystem interactions into the filesystem service.
/**
 * A service class for managing file system interactions related to settings.
 */
class SettingsService {
    /**
     * Constructs the settings service which stores app settings.
     */
    constructor(data_directory) {
        this.data_directory = data_directory;
        this.cached_settings = undefined;
        this.settings_directory = path.join(this.data_directory, 'sfvc_settings.json');
    }
    getDefaultSettings() {
        return {
            captures_directory: path.join(this.data_directory, 'Captures'),
            plugins_directory: path.join(this.data_directory, 'Plugins'),
            capture_format: "capture_{id}_{time}_{date}",
            event_format: "event_{id}_{time}_{date}"
        };
    }
    async getPluginsDirectory() {
        return (await (this.getSettings())).plugins_directory;
    }
    async getCapturesDirectory() {
        return (await (this.getSettings())).captures_directory;
    }
    async reloadSettings() {
        this.cached_settings = undefined;
        this.getSettings();
    }
    /**
     * Loads application settings from the JSON file.
     *
     * - If the settings file is missing, a new one is created with default values.
     * - If the file contains invalid or corrupted data, it is renamed with a timestamp suffix and replaced with a new default file.
     * - After the first successful load, settings are cached in memory and returned from there in subsequent calls.
     *
     * @returns A promise resolving to the parsed application settings.
     */
    async getSettings() {
        // Return cached settings if already loaded
        if (this.cached_settings)
            return this.cached_settings;
        if (!this.filesystem) {
            const defaults = this.getDefaultSettings();
            this.cached_settings = defaults;
            return defaults;
        }
        return this.filesystem.getSettings(this.settings_directory);
    }
    /**
     * Saves application settings to the JSON file.
     * @param settings - The settings object to save.
     */
    async saveSettings() {
        if (!this.filesystem) {
            throw new Error("Could not save as filesystem has not been registered. Did you forget to call registerFilesystem(filesystem) on this settings service?");
        }
        if (!this.cached_settings) {
            this.cached_settings = this.getDefaultSettings();
        }
        await this.filesystem?.saveSettings(this.settings_directory, this.cached_settings);
    }
    /**
     * Attempts to set AppSettings, returns caught errors.
     * @param settings new Settings.
     * @returns ZodIssues if there were errors, empty Array otherwise.
     */
    setSettingsWithError(settings) {
        try {
            const validated = schemas_1.AppSettingsSchema.parse(settings);
            this.cached_settings = validated;
            return [];
        }
        catch (error) {
            if (error instanceof zod_1.ZodError) {
                return error.errors;
            }
            else {
                throw error;
            }
        }
    }
    registerFilesystem(filesystem) {
        this.filesystem = filesystem;
    }
}
exports.default = SettingsService;
