"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/** @inheritdoc */
class BasenameLookupService {
    /**
     * Creates a new BasenameLookupService which will use the given filesystem to build its internal database.
     * @param filesystem The filesystem to use when populating the internal database.
     * @param logger The logger where any errors will be logged.
     */
    constructor(filesystem, logger) {
        this.filesystem = filesystem;
        this.logger = logger;
        /**
         * Capture ID -> Event ID -> Event Basename
         */
        this.event_basenames = new Map();
        /**
         * Capture ID -> Capture Basename
         */
        this.capture_basenames = new Map();
    }
    /**
     * Ensures the capture found at the given basename matches the expected capture id.
     * @param capture_id The capture id to check.
     * @param capture_basename The capture basename to check.
     * @returns Whether or not they match.
     */
    async ensureCaptureIdMatch(capture_id, capture_basename) {
        try {
            const capture = await this.filesystem.getCapture(capture_basename);
            return capture.capture_id === capture_id;
        }
        catch {
            return false;
        }
        ;
    }
    /**
     * Ensures the capture and event found at the given basenames matches the expected capture id and event id.
     * @param capture_id The capture id to check.
     * @param capture_basename The capture basename to check.
     * @param event_id The event id to check.
     * @param event_basename The event basename to check.
     * @returns Whether or not they match.
     */
    async ensureEventIdMatch(capture_id, capture_basename, event_id, event_basename) {
        if (await this.ensureCaptureIdMatch(capture_id, capture_basename)) {
            try {
                const event = await this.filesystem.getEvent(capture_basename, event_basename);
                return event.manifest.event_id === event_id;
            }
            catch {
                return false;
            }
            ;
        }
        return false;
    }
    /** @inheritdoc */
    async getEventBasename(capture_id, event_id) {
        const capture_basename = await this.getCaptureBasename(capture_id);
        if (capture_basename) {
            const event_basename_map = this.event_basenames.get(capture_id);
            if (event_basename_map) {
                const event_basename = event_basename_map.get(event_id);
                if (event_basename) {
                    if (await this.ensureEventIdMatch(capture_id, capture_basename, event_id, event_basename)) {
                        return event_basename;
                    }
                    else {
                        event_basename_map.delete(event_id);
                    }
                }
            }
            const event_basenames = await this.filesystem.listEventBasenames(capture_basename);
            for (const current_event_basename of event_basenames) {
                const event = await this.filesystem.getEvent(capture_basename, current_event_basename);
                if (event_id === event.manifest.event_id) {
                    if (event_basename_map) {
                        event_basename_map.set(event_id, current_event_basename);
                        return current_event_basename;
                    }
                    this.event_basenames.set(capture_id, new Map([[event_id, current_event_basename]]));
                    return current_event_basename;
                }
            }
        }
        return undefined;
    }
    /** @inheritdoc */
    async getCaptureBasename(capture_id) {
        let capture_basename = this.capture_basenames.get(capture_id);
        if (capture_basename) {
            if (await this.ensureCaptureIdMatch(capture_id, capture_basename)) {
                return capture_basename;
            }
        }
        const capture_basenames = await this.filesystem.listCaptureBasenames();
        for (const current_capture_basename of capture_basenames) {
            const capture = await this.filesystem.getCapture(current_capture_basename);
            if (capture_id === capture.capture_id) {
                capture_basename = current_capture_basename;
                break;
            }
        }
        return capture_basename;
    }
    /**
     * Populates the database with captures and events using the passed filesystem.
     */
    async populateBasenames() {
        const capture_basenames = await this.filesystem.listCaptureBasenames();
        for (const capture_basename of capture_basenames) {
            const event_basenames = await this.filesystem.listEventBasenames(capture_basename);
            const capture = await this.filesystem.getCapture(capture_basename);
            if (!this.capture_basenames.has(capture.capture_id)) {
                this.capture_basenames.set(capture.capture_id, capture_basename);
            }
            else {
                this.logger.warn(`Duplicate capture ID: ${capture.capture_id}, found basename ${capture_basename} had basename ${this.capture_basenames.get(capture.capture_id)} skipping...`);
            }
            for (const event_basename of event_basenames) {
                const event_basename_map = this.event_basenames.get(capture_basename);
                const event = await this.filesystem.getEvent(capture_basename, event_basename);
                if (event_basename_map) {
                    if (!event_basename_map.has(event.manifest.event_id)) {
                        event_basename_map.set(event.manifest.event_id, event_basename);
                    }
                    else {
                        this.logger.warn(`Duplicate event ID: ${event.manifest.event_id}, found basename ${event_basename} had basename ${event_basename_map.get(event.manifest.event_id)} skipping...`);
                    }
                }
                else {
                    this.event_basenames.set(capture.capture_id, new Map([[event.manifest.event_id, event_basename]]));
                }
            }
        }
    }
}
exports.default = BasenameLookupService;
//# sourceMappingURL=basename_lookup_service.js.map