"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.TraitSet = void 0;
class TraitSet {
    constructor() {
        this.pendingTraits = [];
        this.processedTraits = [];
        this.traitSet = new Set();
    }
    /**
     * Checks if the TraitSet contains this trait already.
     * @param trait The trait to check
     */
    hasTrait(trait) {
        return this.traitSet.has(trait);
    }
    /**
     * Returns true as long as this traitset has unprocessed traits.
     */
    hasPending() {
        return this.pendingTraits.length > 0;
    }
    /**
     * Adds a trait from this TraitSet if it is not already present.
     * @param trait The trait to add
     */
    addTrait(trait) {
        if (!this.traitSet.has(trait)) {
            this.traitSet.add(trait);
            this.pendingTraits.push(trait);
        }
    }
    /**
     * Returns the next pending trait but does not advance process head
     * @returns The next pending trait on the trait set
     */
    peekNextPending() {
        return this.pendingTraits.length > 0 ? this.pendingTraits[0] : "";
    }
    /**
     * Returns last processed trait.
     * @returns Last processed trait
     */
    peekLastProcessed() {
        return this.processedTraits.length > 0 ? this.processedTraits[this.processedTraits.length - 1] : "";
    }
    /**
     * Returns the next pending trait, immediately moves process head to next trait
     * @returns The next pending trait on the trait set
     */
    popNextPending() {
        const next = this.pendingTraits.shift();
        if (next) {
            this.processedTraits.push(next);
            return next;
        }
        return "";
    }
    /**
     * Clears the trait set and the processed/pending traits.
     */
    clearTraits() {
        this.traitSet.clear();
        this.pendingTraits = [];
        this.processedTraits = [];
    }
}
exports.TraitSet = TraitSet;
