
import { Operator, OperatorContext, ILogger, ResultMap } from "#sfvc_types/plugin";
import { ElementHandle, Page } from "@puppeteer";

/**
 * Set up the YT-Shorts Operator
 * @param page Reference to the page used for this Operator
 */
export async function activate(context: OperatorContext, page: Page): Promise<Operator> {
    const logger = context.logger;
    logger.debug("activated yt-shorts operator")
    const page_url = 'https://www.youtube.com/shorts';

    // Set screen size.
    await page.setViewport({ width: 1080, height: 1024 });

    await page.goto(page_url);

    let slider: ElementHandle<Element> | undefined = undefined;

    const playerAriaPath = 'aria/YouTube Video Player';
    const seekerAriaPath = 'aria/Seek slider';
    logger.warn("ACTION REQUIRED: Please accept or reject the cookies prompt manually to initiate scraping.")
    //Wait for the researcher to accept prompts/reject cookies or whatever is necessary to get the app into a working state.
    await page.waitForSelector(playerAriaPath);
    // Click on video to start playing.
    await page.locator(playerAriaPath).click();
    // Grab reference to seek slider.
    slider = await page.locator(seekerAriaPath).waitHandle();

    return {
        scroll: async () => {
            await page.keyboard.press('ArrowDown');

            // Optional: wait a bit for content to load after scrolling
            await page.evaluate(RandomPromise);
        },
        operations: {
            "id": {
                run: async (currentResultMap: ResultMap) => {
                    const id = new URL(page.url()).pathname.split('/').filter(Boolean).pop();
                    currentResultMap.set("id", { raw: "", processed: id! })
                    logger.debug(`id: ${id}`)
                }
            },
            "comments": {
                run: async (currentResultMap: ResultMap) => {
                    // gather comments with Puppeteer, add ResultMap to map
                    const comments = await extractComments(context, page, logger);
                    currentResultMap.set("comments",
                        { raw: "", processed: comments })
                    logger.warn('still returning comments as flat string')
                    logger.debug(`comments: ${comments}`)
                },
                cleanup: async function (): Promise<void> {
                    await page.keyboard.press('Escape');
                }
            },
            "url": {
                run: async (currentResultMap: ResultMap) => {
                    currentResultMap.set("url", { raw: "", processed: page.url() })
                    logger.debug(`url: ${page.url()}`)
                }
            },
            "creator": {
                run: async (currentResultMap: ResultMap) => {
                    const creatorEl = (await page.$(".ytReelChannelBarViewModelChannelName"))?.asElement()
                    const creator = await creatorEl?.evaluate(el => el.textContent)
                    if (!creator) logger.warn("no creator found")
                    currentResultMap.set("creator", { raw: "", processed: creator as string })
                    logger.debug(`creator: ${creator}`)
                }
            },
            "created_at": {
                run: async (currentResultMap: ResultMap) => {
                    //When this runs the description pane will already be open (see 'depends_on')
                    let factoids = await page.$$("factoid-renderer")
                    if (factoids.length == 0) {
                        logger.warn("no factoids found, looking for 'created_at'")
                        currentResultMap.set("created_at", { raw: "", processed: "" })
                        return
                    }
                    let createdAtValue = await page.evaluate(() => {
                        const els = document.querySelectorAll(".ytwFactoidRendererFactoid")
                        const value = els[2].getAttribute("aria-label")
                        return value
                    }, factoids);
                    if (createdAtValue?.includes("ago")) logger.warn("handle the case where the value is a relative time value e.g. '20 minutes ago'")
                    const createdAt = createdAtValue ? createdAtValue : "undefined"
                    currentResultMap.set("created_at", { raw: "", processed: createdAt as string })
                    logger.debug(`created_at: ${createdAt}`)
                },
                depends_on: "description"
            },
            "thumbnail_url": {
                run: async (currentResultMap: ResultMap) => {
                    //TODO implement
                    logger.warn(`trait 'thumbnail_url' not implemented`)
                }
            },
            "title": {
                run: async (currentResultMap: ResultMap) => {
                    const titleEl = (await page.$(".ytShortsVideoTitleViewModelHostClickable"))?.asElement()
                    const title = await titleEl?.evaluate(el => el.textContent)
                    if (!title) logger.warn("no title found")
                    currentResultMap.set("title", { raw: "", processed: title as string })
                    logger.debug(`title: ${title}`)
                }
            },
            "description": {
                run: async (currentResultMap: ResultMap) => {
                    await openDescription(context, page, logger)
                    let description = await page.evaluate((selector) => {
                        const els = document.querySelectorAll(selector)
                        const value = els[0].textContent
                        return value
                    }, "ytd-expandable-video-description-body-renderer");
                    if (!description) logger.warn("no description found")
                    currentResultMap.set("description", { raw: "", processed: description as string })
                    logger.debug(`description: ${(description as string).trim()}`)
                },
                cleanup: async () => {
                    await page.keyboard.press('Escape');
                },
            },
            "comments_count": {
                run: async (currentResultMap: ResultMap) => {
                    const commentCountEl = await page.$("#comments-button")
                    const commentsCount = await commentCountEl?.evaluate(el => el.textContent)
                    if (!commentsCount) logger.warn("no comments_count found")
                    const cleanedCount = commentsCount?.replace(/Comments?/, "").trim()
                    currentResultMap.set("comments_count", { raw: "", processed: cleanedCount as string })
                    logger.debug(`comments_count: ${cleanedCount}`)
                }
            },
            "likes_count": {
                run: async (currentResultMap: ResultMap) => {
                    let likesCount = await page.evaluate(() => {
                        const els = document.querySelectorAll(".ytwFactoidRendererFactoid")
                        const value = els[0].getAttribute("aria-label")
                        return value
                    }, "#factoids");
                    if (!likesCount) logger.warn("no likes count found")
                    currentResultMap.set("likes_count", { raw: "", processed: likesCount as string })
                    logger.debug(`likes_count: ${likesCount}`)
                },
                depends_on: "description"
            },
            "views_count": {
                run: async (currentResultMap: ResultMap) => {
                    await openDescription(context, page, logger)
                    let viewsCount = await page.evaluate(() => {
                        const els = document.querySelectorAll(".ytwFactoidRendererFactoid")
                        const value = els[1].getAttribute("aria-label")
                        return value
                    }, "#factoids");
                    if (!viewsCount) logger.warn("no views count found")
                    currentResultMap.set("views_count", { raw: "", processed: viewsCount as string })
                    logger.debug(`views_count: ${viewsCount}`)
                }
            }
        }
    };
}

// opens the description panel through the "More Actions" button
async function openDescription(ctx: OperatorContext, page: Page, logger: ILogger) {
    // TODO: check if the description is already opened

    // first click on the three-dot menu
    const menuButtonSelector = '#menu-button button';
    await page.waitForSelector(menuButtonSelector, { timeout: 5000 });
    await page.evaluate((selector) => {
        const el = document.querySelector(selector);
        (el as HTMLElement).click()
    }, menuButtonSelector);

    const moreMenu = (await page.$("tp-yt-paper-listbox"))
    if (!moreMenu) {
        logger.warn('could not find the more menu button')
        return
    }

    // the first item of the menu is the description
    const descriptionButtonSelector = "ytd-menu-service-item-renderer"
    await page.waitForSelector(descriptionButtonSelector, { timeout: 5000 });
    const descriptionButton = (await page.$(descriptionButtonSelector))
    if (!descriptionButton) {
        logger.warn('could not find the description button')
        return
    }
    await page.evaluate((selector) => {
        const el = document.querySelectorAll(selector)[0];
        (el as HTMLElement).click()
    }, descriptionButtonSelector);

    logger.debug("description panel opened")
}

async function extractComments(ctx: OperatorContext, page: Page, logger: ILogger): Promise<string> {

    const commentsButtonSelector = 'button[aria-label^="View"][aria-label$="comments"]';
    const commentThreadSelector = 'ytd-comment-thread-renderer';

    // Open comments
    await page.waitForSelector(commentsButtonSelector, { timeout: 5000 });
    await page.click(commentsButtonSelector);

    // Wait for comments to load
    await page.waitForSelector(commentThreadSelector, { timeout: 5000 });
    await page.evaluate(RandomPromise())

    const count = await page.evaluate(() => {
        const els = document.querySelectorAll('ytd-comment-thread-renderer');
        return els.length;
    });

    const samples = await page.$$eval('ytd-comment-thread-renderer', els =>
        els.map(el => ({
            expanderCount: el.querySelectorAll('ytd-expander').length,
            contentSnippet: el.querySelector('yt-attributed-string#content-text span')?.textContent?.trim().slice(0, 100) ?? ''
        }))
    );
    let comments: Array<string> = []

    for (const sample of samples) {
        comments.push(sample.contentSnippet)
    }

    // Scroll down a bit to help lazy-load
    await page.evaluate(() => window.scrollBy(0, window.innerHeight * 0.9));
    await page.evaluate(RandomPromise())


    // Close the comments box
    const closeButtonSelector = "#visibility-button"
    await page.click(closeButtonSelector);
    await page.evaluate(RandomPromise()) // wait briefly for the panel to collapse

    return JSON.stringify({ comments });
}

function RandomPromise() {
    return () => new Promise(r => setTimeout(r, 1000 + (Math.random() * 1000)));
}